/* Copyright (C) 2001 Roberto Bagnara <bagnara@cs.unipr.it>

This file is free software; as a special exception the author gives
unlimited permission to copy and/or distribute it, with or without
modifications, as long as this notice is preserved.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY, to the extent permitted by law; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE. */

#ifndef _Variable_types_hh
#define _Variable_types_hh 1

namespace Parma_Polyhedra_Library {
  class Variable;
}

#endif
/* Copyright (C) 2001 Roberto Bagnara <bagnara@cs.unipr.it>

This file is free software; as a special exception the author gives
unlimited permission to copy and/or distribute it, with or without
modifications, as long as this notice is preserved.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY, to the extent permitted by law; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE. */

#ifndef _LinExpression_types_hh
#define _LinExpression_types_hh 1

namespace Parma_Polyhedra_Library {
  class LinExpression;
}

#endif
/* Copyright (C) 2001 Roberto Bagnara <bagnara@cs.unipr.it>

This file is free software; as a special exception the author gives
unlimited permission to copy and/or distribute it, with or without
modifications, as long as this notice is preserved.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY, to the extent permitted by law; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE. */

#ifndef _Constraint_types_hh
#define _Constraint_types_hh 1

namespace Parma_Polyhedra_Library {
  class Constraint;
}

#endif
/* Copyright (C) 2001 Roberto Bagnara <bagnara@cs.unipr.it>

This file is free software; as a special exception the author gives
unlimited permission to copy and/or distribute it, with or without
modifications, as long as this notice is preserved.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY, to the extent permitted by law; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE. */

#ifndef _ConSys_types_hh
#define _ConSys_types_hh 1

namespace Parma_Polyhedra_Library {
  class ConSys;
}

#endif
/* Copyright (C) 2001 Roberto Bagnara <bagnara@cs.unipr.it>

This file is free software; as a special exception the author gives
unlimited permission to copy and/or distribute it, with or without
modifications, as long as this notice is preserved.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY, to the extent permitted by law; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE. */

#ifndef _Generator_types_hh
#define _Generator_types_hh 1

namespace Parma_Polyhedra_Library {
  class Generator;
}

#endif
/* Copyright (C) 2001 Roberto Bagnara <bagnara@cs.unipr.it>

This file is free software; as a special exception the author gives
unlimited permission to copy and/or distribute it, with or without
modifications, as long as this notice is preserved.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY, to the extent permitted by law; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE. */

#ifndef _GenSys_types_hh
#define _GenSys_types_hh 1

namespace Parma_Polyhedra_Library {
  class GenSys;
}

#endif
/* Copyright (C) 2001 Roberto Bagnara <bagnara@cs.unipr.it>

This file is free software; as a special exception the author gives
unlimited permission to copy and/or distribute it, with or without
modifications, as long as this notice is preserved.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY, to the extent permitted by law; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE. */

#ifndef _Polyhedron_types_hh
#define _Polyhedron_types_hh 1

//! The entire library is confined into this namespace.
namespace Parma_Polyhedra_Library {
  class Polyhedron;
}

#endif
/* Variable class declaration.
   Copyright (C) 2001 Roberto Bagnara <bagnara@cs.unipr.it>

This file is part of the Parma Polyhedra Library (PPL).

The PPL is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

The PPL is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
USA.

For the most up-to-date information see the Parma Polyhedra Library
site: http://www.cs.unipr.it/ppl/ . */

#ifndef _Variable_defs_hh
#define _Variable_defs_hh 1

#include "Variable.types.hh"
#include <iosfwd>

//! A dimension of the space.
/*!
  An object of the class Variable represents a dimension of the space,
  that is one of the Cartesian axes.
  Variables are used as base blocks in order to build
  more complex linear expressions.
  Each variable is identified by a non-negative integer,
  representing the index of the corresponding Cartesian axis
  (the first axis has index 0).

  Note that the ``meaning'' of an object of the class Variable
  is completely specified by the integer index provided to its
  constructor:
  be careful not to be mislead by C++ language variable names.
  For instance, in the following example the linear expressions
  <CODE>e1</CODE> and <CODE>e2</CODE> are equivalent,
  since the two variables <CODE>x</CODE> and <CODE>z</CODE> denote
  the same Cartesian axis.
  \code
  Variable x(0);
  Variable y(1);
  Variable z(0);
  LinExpression e1 = x + y;
  LinExpression e2 = y + z;
  \endcode

*/

class Parma_Polyhedra_Library::Variable {

public:
  //! Constructor: \p id is the index of the Cartesian axis.
  explicit Variable(unsigned int id);
  //! Returns the index of the Cartesian axis.
  unsigned int id() const;

private:
  //! The index of the Cartesian axis.
  unsigned int varid;
};

namespace Parma_Polyhedra_Library {

  //! Output operator.
  /*! \relates Variable */
  std::ostream&
  operator <<(std::ostream& s, const Parma_Polyhedra_Library::Variable& v);

  //! Defines a total ordering on variables.
  /*! \relates Variable */
  bool operator <(const Variable& v, const Variable& w);
}

#include "Variable.inlines.hh"

#endif
/* LinExpression class declaration.
   Copyright (C) 2001 Roberto Bagnara <bagnara@cs.unipr.it>

This file is part of the Parma Polyhedra Library (PPL).

The PPL is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

The PPL is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
USA.

For the most up-to-date information see the Parma Polyhedra Library
site: http://www.cs.unipr.it/ppl/ . */

#ifndef _LinExpression_defs_hh
#define _LinExpression_defs_hh 1

#include "LinExpression.types.hh"
#include "Row.defs.hh"
#include "Variable.types.hh"
#include <cstddef>

namespace Parma_Polyhedra_Library {
  // Put them in the namespace here to declare them friend later.
  LinExpression operator +(const LinExpression& e1, const LinExpression& e2);
  LinExpression operator +(const Integer& n, const LinExpression& e);
  LinExpression operator +(const LinExpression& e, const Integer& n);

  LinExpression operator -(const LinExpression& e);

  LinExpression operator -(const LinExpression& e1, const LinExpression& e2);
  LinExpression operator -(const Integer& n, const LinExpression& e);
  LinExpression operator -(const LinExpression& e, const Integer& n);

  LinExpression operator *(const Integer& n, const LinExpression& e);
  LinExpression operator *(const LinExpression& e, const Integer& n);

  LinExpression& operator +=(LinExpression& e1, const LinExpression& e2);
  LinExpression& operator +=(LinExpression& e, const Variable& v);
  LinExpression& operator +=(LinExpression& e, const Integer& n);
}

//! A linear expression.
/*!
    An object of the class LinExpression represents the linear expression
    \f[
      \sum_{i=0}^{n-1} a_i x_i + b
    \f]
    where \f$n\f$ is the dimension of the space,
    each \f$a_i\f$ is the integer coefficient
    of the \p i -th variable \f$x_i\f$
    and \f$b\f$ is the integer for the inhomogeneous term.

    \par How to build a linear expression.

    Linear expressions are the basic blocks for defining
    both constraints (i.e., linear equalities or inequalities)
    and generators (i.e., lines, rays and vertices).
    A full set of functions is defined to provide a convenient interface
    for building complex linear expressions starting from simpler ones
    and from objects of the classes Variable and Integer:
    available operators include unary negation,
    binary addition and subtraction,
    as well as multiplication by an Integer.
    The space-dimension of a linear expression is defined as the maximum
    space-dimension of the arguments used to build it:
    in particular, the space-dimension of a Variable <CODE>x</CODE>
    is defined as <CODE>x.id()+1</CODE>,
    whereas all the objects of the class Integer have space-dimension zero.

    \par Example
    The following code builds the linear expression \f$4x - 2y - z + 14\f$,
    having space-dimension \f$3\f$:
    \code
  LinExpression e = 4*x - 2*y - z + 14;
    \endcode
    Another way to build the same linear expression is:
    \code
  LinExpression e1 = 4*x;
  LinExpression e2 = 2*y;
  LinExpression e3 = z;
  LinExpression e = LinExpression(14);
  e += e1 - e2 - e3;
    \endcode
    Note that \p e1, \p e2 and \p e3 have space-dimension 1, 2 and 3,
    respectively; also, in the fourth line of code, \p e is created
    with space-dimension zero and then extended to space-dimension 3.
*/

class Parma_Polyhedra_Library::LinExpression : private Row {

public:
  //! Default constructor: returns a copy of LinExpression::zero().
  LinExpression();
  //! Ordinary copy-constructor.
  LinExpression(const LinExpression& e);
  //! Destructor.
  virtual ~LinExpression();

  //! Constructor: builds the linear expression corresponding
  //! to the inhomogeneous term \p n.
  explicit LinExpression(const Integer& n);
  //! Constructor: builds the linear expression corresponding
  //! to the variable \p v.
  LinExpression(const Variable& v);

  //! Returns the dimension of the vector space enclosing \p *this.
  size_t space_dimension() const;

  //! Returns the (zero-dimension space) constant 0.
  static const LinExpression& zero();

private:
  //! Copy-constructor with a specified dimension.
  LinExpression(const LinExpression& e, size_t size);

private:
  //! Implementation sizing constructor.
  //! The bool parameter is just to avoid problems with
  //! the constructor LinExpression(const Integer& n).
  LinExpression(size_t size, bool);

  friend class Constraint;
  friend class Generator;

  //! Returns the linear expression \p e1 + \p e2.
  friend LinExpression
  Parma_Polyhedra_Library::operator +(const LinExpression& e1,
				      const LinExpression& e2);

  //! Returns the linear expression \p n + \p e.
  friend LinExpression
  Parma_Polyhedra_Library::operator +(const Integer& n,
				      const LinExpression& e);

  //! Returns the linear expression \p e + \p n.
  friend LinExpression
  Parma_Polyhedra_Library::operator +(const LinExpression& e,
 				      const Integer& n);

  //! Returns the linear expression - \p e.
  friend LinExpression
  Parma_Polyhedra_Library::operator -(const LinExpression& e);

  //! Returns the linear expression \p e1 - \p e2.
  friend LinExpression
  Parma_Polyhedra_Library::operator -(const LinExpression& e1,
				      const LinExpression& e2);

  //! Returns the linear expression \p n - \p e.
  friend LinExpression
  Parma_Polyhedra_Library::operator -(const Integer& n,
				      const LinExpression& e);

  //! Returns the linear expression \p e - \p n.
  friend LinExpression
  Parma_Polyhedra_Library::operator -(const LinExpression& e,
				      const Integer& n);

  //! Returns the linear expression \p n * \p e.
  friend LinExpression
  Parma_Polyhedra_Library::operator *(const Integer& n,
				      const LinExpression& e);

  //! Returns the linear expression \p e * \p n.
  friend LinExpression
  Parma_Polyhedra_Library::operator *(const LinExpression& e,
				      const Integer& n);

  //! Returns the linear expression \p e1 + \p e2 and assigns it to \p e1.
  friend LinExpression&
  Parma_Polyhedra_Library::operator +=(LinExpression& e1,
				       const LinExpression& e2);

  //! Returns the linear expression \p e + \p v and assigns it to \p e.
  friend LinExpression&
  Parma_Polyhedra_Library::operator +=(LinExpression& e,
				       const Variable& v);

  //! Returns the linear expression \p e + \p n and assigns it to \p e.
  friend LinExpression&
  Parma_Polyhedra_Library::operator +=(LinExpression& e,
                                       const Integer& n);
};

#include "LinExpression.inlines.hh"

#endif




/* Constraint class declaration.
   Copyright (C) 2001 Roberto Bagnara <bagnara@cs.unipr.it>

This file is part of the Parma Polyhedra Library (PPL).

The PPL is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

The PPL is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
USA.

For the most up-to-date information see the Parma Polyhedra Library
site: http://www.cs.unipr.it/ppl/ . */

#ifndef _Constraint_defs_hh
#define _Constraint_defs_hh 1

#include "Constraint.types.hh"
#include "Row.defs.hh"
#include "Variable.defs.hh"
#include "LinExpression.types.hh"
#include <iosfwd>

namespace Parma_Polyhedra_Library {

  //! Output operator.
  /*! \relates Constraint */
  std::ostream& operator <<(std::ostream& s, const Constraint& c);

  // Put them in the namespace here to declare them friend later.
  Constraint operator ==(const LinExpression& e1, const LinExpression& e2);
  Constraint operator ==(const LinExpression& e, const Integer& n);
  Constraint operator ==(const Integer& n, const LinExpression& e);

  Constraint operator <=(const LinExpression& e1, const LinExpression& e2);
  Constraint operator <=(const LinExpression& e, const Integer& n);
  Constraint operator <=(const Integer& n, const LinExpression& e);

  Constraint operator >=(const LinExpression& e1, const LinExpression& e2);
  Constraint operator >=(const LinExpression& e, const Integer& n);
  Constraint operator >=(const Integer& n, const LinExpression& e);

  Constraint operator >>(const Constraint& c, unsigned int offset);
}

//! A linear equality or inequality.
/*!
  An object of the class Constraint is either:
  - an equality: \f$\sum_{i=0}^{n-1} a_i x_i + b = 0\f$; or
  - an inequality: \f$\sum_{i=0}^{n-1} a_i x_i + b \geq 0\f$;

  where \f$n\f$ is the dimension of the space.

  \par How to build a constraint
  Constraints are typically built by applying a relational operator
  to a pair of linear expressions.
  Available relational operators include equality (<CODE>==</CODE>)
  and non-strict inequalities (<CODE>>=</CODE> and <CODE><=</CODE>).
  Strict inequalities (<CODE><</CODE> and <CODE>></CODE>)
  are not supported.
  The space-dimension of a constraint is defined as the maximum
  space-dimension of the arguments of its constructor.

  \par
  In the following example it is assumed that variables
  <CODE>x</CODE>, <CODE>y</CODE> and <CODE>z</CODE>
  are defined as follows:
  \code
  Variable x(0);
  Variable y(1);
  Variable z(2);
  \endcode

  \par Example
  The following code builds the equality constraint
  \f$3x + 5y - z = 0\f$, having space-dimension \f$3\f$:
  \code
  Constraint eq_c(3*x + 5*y - z == 0);
  \endcode
  The following code builds the inequality constraint
  \f$4x \geq 2y - 13\f$, having space-dimension \f$2\f$:
  \code
  Constraint ineq_c(4*x >= 2*y - 13);
  \endcode
  The unsatisfiable constraint on the zero-dimension space \f$\Rset^0\f$
  can be specified as follows:
  \code
  Constraint false_c = Constraint::zero_dim_false();
  \endcode
  An equivalent, but more involved way is the following:
  \code
  Constraint false_c(LinExpression::zero() == 1);
  \endcode
  In constrast, the following code defines an unsatisfiable constraint
  having space-dimension \f$3\f$:
  \code
  Constraint false_c(0*z == 1);
  \endcode

*/
class Parma_Polyhedra_Library::Constraint : private Row {
private:
  //! Default constructor: private and not implemented.
  Constraint();

  Constraint(LinExpression& e);

  Constraint(Row::Type type, size_t size);

  //! Returns the constraint \p e1 = \p e2.
  friend Constraint
  Parma_Polyhedra_Library::operator ==(const LinExpression& e1,
				       const LinExpression& e2);

  //! Returns the constraint \p e = \p n.
  friend Constraint
  Parma_Polyhedra_Library::operator ==(const LinExpression& e,
				       const Integer& n);

  //! Returns the constraint \p n = \p e.
  friend Constraint
  Parma_Polyhedra_Library::operator ==(const Integer& n,
				       const LinExpression& e);

  //! Returns the constraint \p e1 >= \p e2.
  friend Constraint
  Parma_Polyhedra_Library::operator >=(const LinExpression& e1,
				       const LinExpression& e2);

  //! Returns the constraint \p e >= \p n.
  friend Constraint
  Parma_Polyhedra_Library::operator >=(const LinExpression& e,
				       const Integer& n);

  //! Returns the constraint \p n >= \p e.
  friend Constraint
  Parma_Polyhedra_Library::operator >=(const Integer& n,
				       const LinExpression& e);

  //! Returns the constraint \p e1 <= \p e2.
  friend Constraint
  Parma_Polyhedra_Library::operator <=(const LinExpression& e1,
				       const LinExpression& e2);

  //! Returns the constraint \p e <= \p n.
  friend Constraint
  Parma_Polyhedra_Library::operator <=(const LinExpression& e,
				       const Integer& n);

  //! Returns the constraint \p n <= \p e.
  friend Constraint
  Parma_Polyhedra_Library::operator <=(const Integer& n,
				       const LinExpression& e);

  //! Returns the constraint \p c with variables renamed
  //! by adding \p offset to their Cartesian axis identifier.
  friend Constraint
  Parma_Polyhedra_Library::operator >>(const Constraint& c,
				       unsigned int offset);

public:
  //! Ordinary copy-constructor.
  Constraint(const Constraint& c);
  //! Destructor.
  ~Constraint();

  //! Returns <CODE>true</CODE> if and only if
  //! \p *this is an equality constraint.
  bool is_equality() const;
  //! Returns <CODE>true</CODE> if and only if
  //! \p *this is an inequality constraint.
  bool is_inequality() const;

  //! Returns the dimension of the vector space enclosing \p *this.
  size_t space_dimension() const;

  //! Returns the coefficient of \p v in \p *this.
  const Integer& coefficient(Variable v) const;
  //! Returns the inhomogeneous term of \p *this.
  const Integer& coefficient() const;

  //! The unsatisfiable (zero-dimension space) constraint \f$0 = 1\f$.
  static const Constraint& zero_dim_false();

  //! The true (zero-dimension space) constraint \f$0 \leq 1\f$,
  //! also known as <EM>positivity constraint</EM>.
  static const Constraint& zero_dim_positivity();

private:
  //! Returns <CODE>true</CODE> if and only if
  //! \p *this is the trivial true constraint \f$0 <= n\f$,
  //! where \f$n \geq 0\f$.
  bool is_trivial_true() const;

  //! Returns <CODE>true</CODE> if and only if
  //! \p *this is the trifial false constraint
  //! (i.e., either \f$0 >= n\f$, where \f$n > 0\f$
  //! or \f$0 = n\f$, where \f$n \neq 0\f$.
  bool is_trivial_false() const;

  enum Type {
    EQUALITY = Row::LINE_OR_EQUALITY,
    INEQUALITY = Row::RAY_OR_VERTEX_OR_INEQUALITY
  };

  //! Returns the constraint type of \p *this.
  Type type() const;
  //! Sets the constraint type to <CODE>EQUALITY</CODE>.
  void set_is_equality();
  //! Sets the constraint type to <CODE>INEQUALITY</CODE>.
  void set_is_inequality();
};

namespace std {

#ifdef PPL_DOXYGEN_INCLUDE_IMPLEMENTATION_DETAILS
//! Specialize <CODE>std::swap</CODE>.
#endif // PPL_DOXYGEN_INCLUDE_IMPLEMENTATION_DETAILS
void swap(Parma_Polyhedra_Library::Constraint& x,
	  Parma_Polyhedra_Library::Constraint& y);

} // namespace std


#include "Constraint.inlines.hh"

#endif
/* ConSys class declaration.
   Copyright (C) 2001 Roberto Bagnara <bagnara@cs.unipr.it>

This file is part of the Parma Polyhedra Library (PPL).

The PPL is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

The PPL is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
USA.

For the most up-to-date information see the Parma Polyhedra Library
site: http://www.cs.unipr.it/ppl/ . */

#ifndef _ConSys_defs_hh
#define _ConSys_defs_hh 1

#include "LinExpression.defs.hh"
#include "ConSys.types.hh"
#include "Matrix.defs.hh"
#include "Generator.types.hh"
#include "Constraint.defs.hh"
#include <cstddef>
#include <vector>
#include <iterator>


//! A system of constraints.
/*!
    An object of the class ConSys is a system of constraints,
    i.e., a multiset of objects of the class Constraint.
    When inserting constraints in a system, dimensions are automatically
    adjusted so that all the constraints in the system are defined
    on the same vector space.

    \par
    In all the examples it is assumed that variables
    <CODE>x</CODE> and <CODE>y</CODE> are defined as follows:
    \code
  Variable x(0);
  Variable y(1);
    \endcode

    \par Example 1
    The following code builds a system of constraints corresponding to
    a square in \f$\Rset^2\f$:
    \code
  ConSys cs;
  cs.insert(x >= 0);
  cs.insert(x <= 3);
  cs.insert(y >= 0);
  cs.insert(y <= 3);
    \endcode
    Note that:
    the constraint system is created with space dimension zero;
    the first and third constraint insertions increases the space
    dimension to \f$1\f$ and \f$2\f$, respectively.

    \par Example 2
    The following code builds a system of constraints corresponding to
    a half-strip in \f$\Rset^2\f$:
    \code
  ConSys cs;
  cs.insert(x >= 0);
  cs.insert(x - y <= 0);
  cs.insert(x - y + 1 >= 0);
    \endcode

    \note
    After inserting a multiset of constraints in a constraint system,
    there are no guarantees that an <EM>exact</EM> copy of them
    can be retrieved:
    in general, only an <EM>equivalent</EM> constraint system
    will be available, where original constraints may have been
    reordered, removed (if they are trivial, duplicate or
    implied by other constraints), linearly combined, etc.
*/
class Parma_Polyhedra_Library::ConSys : private Matrix {
public:
  //! Default constructor: builds an empty system of constraints.
  ConSys();
  //! Builds the singleton system containing only constraint \p c.
  ConSys(const Constraint& c);
  //! Ordinary copy-constructor.
  ConSys(const ConSys& cs);
  //! Destructor.
  virtual ~ConSys();

  //! Assignment operator.
  ConSys& operator =(const ConSys& y);

  //! Returns the dimension of the vector space enclosing \p *this.
  size_t space_dimension() const;

  //! Inserts a copy of the constraint \p c into \p *this,
  //! increasing the number of dimensions if needed.
  void insert(const Constraint& c);

  //! Returns the singleton system containing only
  //! Constraint::zero_dim_false().
  static const ConSys& zero_dim_empty();

  /*!
      A const_iterator is used to provide read-only access
      to each constraint contained in an object of ConSys.

      \par Example
      The following code prints the system of constraints
      defining the polyhedron <CODE>ph</CODE>:
      \code
  const ConSys cs = ph.constraints();
  ConSys::const_iterator iend = cs.end();
  for (ConSys::const_iterator i = cs.begin(); i != iend; ++i)
    cout << *i << endl;
      \endcode
  */
  class const_iterator
    : public std::iterator<std::forward_iterator_tag,
				Constraint,
				void,
				const Constraint*,
                                const Constraint&> {

  private:
    Matrix::const_iterator i;
    const Matrix* csp;

    //! Constructor.
    const_iterator(const Matrix::const_iterator& iter, const ConSys& csys);
    //! \p *this skips to the last non-trivial constraint.
    void skip_forward();

    friend class ConSys;

  public:
    //! Default constructor.
    const_iterator();
    //! Ordinary copy-constructor.
    const_iterator(const const_iterator& y);
    //! Destructor.
    virtual ~const_iterator();
    //! Assignment operator.
    const_iterator& operator =(const const_iterator& y);
    //! Dereference operator.
    const Constraint& operator *() const;
    //! Indirect member selector.
    const Constraint* operator ->() const;
    //! Prefix increment operator.
    const_iterator& operator ++();
    //! Postfix increment operator.
    const_iterator operator ++(int);
    //! Returns <CODE>true</CODE> if and only if
    //! \p *this and \p y are identical.
    bool operator ==(const const_iterator& y) const;
    //! Returns <CODE>true</CODE> if and only if
    //! \p *this and \p y are different.
    bool operator !=(const const_iterator& y) const;
  };

  //! Returns the const_iterator pointing to the first constraint,
  //! if \p *this is not empty;
  //! otherwise, returns the past-the-end const_iterator.
  const_iterator begin() const;
  //! Returns the past-the-end const_iterator.
  const_iterator end() const;

private:
  //! Constructor: it builds a system of \p num_rows constraints
  //! on a \p num_columns - 1 dimensional space.
  ConSys(size_t num_rows, size_t num_columns);

  //! Returns the \p k- th constraint of the system.
  Constraint& operator [](size_t k);
  //! Returns a constant reference to the \p k- th constraint of the system.
  const Constraint& operator [](size_t k) const;

  //! Returns <CODE>true</CODE> if \p g satisfies all the constraints.
  bool satisfies_all_constraints(const Generator& g) const;

  //! Substitutes a given column of coefficients by a given affine expression.
  void affine_preimage(size_t v,
		       const LinExpression& expr,
		       const Integer& denominator);

  //! Returns the number of the equality constraints.
  size_t num_equalities() const;
  //! Returns the number of the inequality constraints.
  size_t num_inequalities() const;

  //! Checks if all the invariants are satisfied.
  bool OK() const;

 private:
  //! Input operator.
  void get(std::istream& s);
  //! Output operator.
  void print(std::ostream& s) const;
};

namespace std {

//! Specialize std::swap.
void swap(Parma_Polyhedra_Library::ConSys& x,
	  Parma_Polyhedra_Library::ConSys& y);

} // namespace std

#include "ConSys.inlines.hh"

#endif
/* Generator class declaration.
   Copyright (C) 2001 Roberto Bagnara <bagnara@cs.unipr.it>

This file is part of the Parma Polyhedra Library (PPL).

The PPL is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

The PPL is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
USA.

For the most up-to-date information see the Parma Polyhedra Library
site: http://www.cs.unipr.it/ppl/ . */

#ifndef _Generator_defs_hh
#define _Generator_defs_hh 1

#include "Generator.types.hh"
#include "Row.defs.hh"
#include "Variable.defs.hh"
#include "LinExpression.defs.hh"
#include <iosfwd>

namespace Parma_Polyhedra_Library {

  //! Output operator.
  /*! \relates Generator */
  std::ostream&
  operator <<(std::ostream& s, const Generator& g);

  // Put them in the namespace here to declare them friend later.
  Generator line(const LinExpression& e);
  Generator ray(const LinExpression& e);
  Generator vertex(const LinExpression& e, const Integer& d);

}

//! A line, ray or vertex.
/*!
  An object of the class Generator is one of the following:

  - a line \f$\vect{l} = (a_0, \ldots, a_{n-1})^\transpose\f$;
	
  - a ray \f$\vect{r} = (a_0, \ldots, a_{n-1})^\transpose\f$;

  - a vertex
    \f$\vect{v} = (\frac{a_0}{d}, \ldots, \frac{a_{n-1}}{d})^\transpose\f$;

  where \f$n\f$ is the dimension of the space.

  \par A note on terminology.
  As observed in the Introduction, there are cases when, in order to
  represent a polyhedron \f$P\f$ using generators, we need to include
  in the finite set \f$V\f$ even points of \f$P\f$ that are <EM>not</EM>
  vertices of \f$P\f$.
  Nonetheless, accordingly to what is now an established terminology,
  we will call <EM>vertex</EM> any element of the set of generators \f$V\f$,
  even though it is not a ``proper'' vertex of \f$P\f$.

  \par How to build a generator.
  Each type of generator is built by applying the corresponding
  function (<CODE>line</CODE>, <CODE>ray</CODE> or <CODE>vertex</CODE>)
  to a linear expression, representing a direction in the space;
  the space-dimension of the generator is defined as the space-dimension
  of the corresponding linear expression.
  Linear expressions used to define a generator should be homogeneous
  (any constant term will be simply ignored).
  When defining a vertex, an optional Integer argument can be used
  as a common <EM>denominator</EM> for all the coefficients occurring
  in the provided linear expression;
  the default value for this argument is 1.

    \par
    In all the following examples it is assumed that variables
    <CODE>x</CODE>, <CODE>y</CODE> and <CODE>z</CODE>
    are defined as follows:
    \code
  Variable x(0);
  Variable y(1);
  Variable z(2);
    \endcode

    \par Example 1
    The following code builds a line with direction \f$x-y-z\f$
    and having space-dimension \f$3\f$:
    \code
  Generator l = line(x - y - z);
    \endcode
    As mentioned above, the constant term of the linear expression
    is not relevant. Thus, the following code has the same effect:
    \code
  Generator l = line(x - y - z + 15);
    \endcode
    By definition, the origin of the space is not a line, so that
    the following code throws an exception:
    \code
  Generator l = line(0*x);
    \endcode


    \par Example 2
    The following code builds a ray with the same direction as the
    line in Example 1:
    \code
  Generator r = ray(x - y - z);
    \endcode
    As is the case for lines, when specifying a ray the constant term
    of the linear expression is not relevant; also, an exception is thrown
    when trying to built a ray from the origin of the space.

    \par Example 3
    The following code builds the vertex
    \f$\vect{v} = (1, 0, 2)^\transpose \in \Rset^3\f$:
    \code
  Generator v = vertex(1*x + 0*y + 2*z);
    \endcode
    The same effect can be obtained by using the following code:
    \code
  Generator v = vertex(x + 2*z);
    \endcode
    Similarly, the origin \f$\vect{0} \in \Rset^3\f$ can be defined
    using either one of the following lines of code:
    \code
  Generator origin3 = vertex(0*x + 0*y + 0*z);
  Generator origin3_alt = vertex(0*z);
    \endcode
    Note however that the following code would have defined
    a different vertex, namely \f$\vect{0} \in \Rset^2\f$:
    \code
  Generator origin2 = vertex(0*y);
    \endcode
    The following two lines of code both define the only vertex
    having space-dimension zero, namely \f$\vect{0} \in \Rset^0\f$.
    In the second case we exploit the fact that the first argument
    of the function <CODE>vertex</CODE> is optional.
    \code
  Generator origin0 = Generator::zero_dim_vertex();
  Generator origin0_alt = vertex();
    \endcode

    \par Example 4
    The vertex \f$\vect{v}\f$ specified in Example 3 above
    can also be obtained with the following code,
    where we provide a non-default value for the second argument
    of the function <CODE>vertex</CODE> (the denominator):
    \code
  Generator v = vertex(2*x + 0*y + 4*z, 2);
    \endcode
    Obviously, the denominator can be usefully exploited to specify
    vertices having some non-integer (but rational) coordinates.
    For instance, the vertex
    \f$\vect{w} = (-1.5, 3.2, 2.1)^\transpose \in \Rset^3\f$
    can be specified by the following code:
    \code
  Generator w = vertex(-15*x + 32*y + 21*z, 10);
    \endcode
    If a zero denominator is provided, an exception is thrown.
*/

class Parma_Polyhedra_Library::Generator : private Row {
private:
  Generator(LinExpression& e);

  //! Returns the (bidirectional) line of direction \p e.
  //! \exception std::invalid_argument thrown if the homogeneous part
  //!                                  of \p e represents the origin
  //!                                  of the vector space.
  friend Generator
  Parma_Polyhedra_Library::line(const LinExpression& e);
  //! Returns the (unidirectional) ray of direction \p e.
  //! \exception std::invalid_argument thrown if the homogeneous part
  //!                                  of \p e represents the origin
  //!                                  of the vector space.
  friend Generator
  Parma_Polyhedra_Library::ray(const LinExpression& e);
  //! Returns the vertex at \p e / \p d
  //! Both \p e and \p d are optional arguments,
  //! with default values LinExpression::zero() and Integer::one(),
  //! respectively.
  //! \exception std::invalid_argument thrown if \p d is zero.
  friend Generator
  Parma_Polyhedra_Library::vertex(const LinExpression& e
				  = LinExpression::zero(),
				  const Integer& d = Integer_one());

public:
  //! Ordinary copy-constructor.
  Generator(const Generator& g);
  //! Destructor.
  ~Generator();

  //! The generator type.
  enum Type {
    LINE = Row::LINE_OR_EQUALITY,
    RAY = Row::RAY_OR_VERTEX_OR_INEQUALITY,
    VERTEX = RAY+1
  };

  //! Returns the generator type of \p *this.
  Type type() const;

  //! Returns the dimension of the vector space enclosing \p *this.
  size_t space_dimension() const;

  //! Returns the coefficient of \p v in \p *this.
  const Integer& coefficient(Variable v) const;
  //! If \p *this is a vertex, returns its divisor.
  //! \exception std::invalid_argument thrown if \p *this is not a vertex.
  const Integer& divisor() const;

  //! Returns the origin of the zero-dimensional space \f$\Rset^0\f$.
  static const Generator& zero_dim_vertex();

  //! Checks if all the invariants are satisfied.
  bool OK() const;

private:
  //! Returns <CODE>true</CODE> if and only if
  //! \p *this is a line.
  bool is_line() const;
  //! Returns <CODE>true</CODE> if and only if
  //! \p *this is either a ray or a vertex.
  bool is_ray_or_vertex() const;
  //! Sets the type to <CODE>LINE</CODE>.
  void set_is_line();
  //! Sets the type to <CODE>RAY</CODE>.
  void set_is_ray_or_vertex();

private:
  //! Default constructor: private and not implemented.
  Generator();
};

namespace std {

#ifdef PPL_DOXYGEN_INCLUDE_IMPLEMENTATION_DETAILS
//! Specialize <CODE>std::swap</CODE>.
#endif // PPL_DOXYGEN_INCLUDE_IMPLEMENTATION_DETAILS
void swap(Parma_Polyhedra_Library::Generator& x,
	  Parma_Polyhedra_Library::Generator& y);

} // namespace std

#include "Generator.inlines.hh"

#endif
/* GenSys class declaration.
   Copyright (C) 2001 Roberto Bagnara <bagnara@cs.unipr.it>

This file is part of the Parma Polyhedra Library (PPL).

The PPL is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

The PPL is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
USA.

For the most up-to-date information see the Parma Polyhedra Library
site: http://www.cs.unipr.it/ppl/ . */

#ifndef _GenSys_defs_hh
#define _GenSys_defs_hh 1

#include "LinExpression.defs.hh"
#include "GenSys.types.hh"
#include "Matrix.defs.hh"
#include "Generator.types.hh"
#include "Constraint.types.hh"
#include <cstddef>
#include <vector>

namespace Parma_Polyhedra_Library {
  //! Describes possible relations between a system of
  //! generators and a given constraint.
  enum GenSys_Con_Rel {
    //! No generator satisfies the given constraint.
    NONE_SATISFIES,
    //! All generators satisfy the given constraint,
    //! but there exists a generator not saturating it
    //! (i.e., a generator does not belong to the hyper-plane
    //! defined by the constraint.)
    ALL_SATISFY,
    //! All generators saturate the given constraint
    //! (i.e., they all belong to the hyper-plane
    //! defined by the constraint.)
    ALL_SATURATE,
    //! Some generators satisfy the given constraint
    //! (i.e., there exists both a generator satisfying the constraint
    //! and another generator which does not satisfy it.)
    SOME_SATISFY
  };

  //! Output operator for GenSys_Con_Rel.
  std::ostream& operator <<(std::ostream& s, GenSys_Con_Rel r);
}

//! A system of generators.
/*!
    An object of the class GenSys is a system of generators,
    i.e., a multiset of objects of the class Generator
    (lines, rays and vertices).
    When inserting generators in a system, dimensions are automatically
    adjusted so that all the generators in the system are defined
    on the same vector space.
    A system of generators which is meant to define a non-empty polyhedron
    must include at least one vertex, even if the polyhedron has
    no ``proper'' vertices: the reason is that lines and rays need
    a supporting point (they only specify directions).

    \par
     In all the examples it is assumed that variables
    <CODE>x</CODE> and <CODE>y</CODE> are defined as follows:
    \code
  Variable x(0);
  Variable y(1);
    \endcode

    \par Example 1
    The following code defines the line having the same direction
    as the \f$x\f$ axis (i.e., the first Cartesian axis)
    in \f$\Rset^2\f$:
    \code
  GenSys gs;
  gs.insert(line(x + 0*y));
    \endcode
    As said above, this system of generators corresponds to
    an empty polyhedron, because the line has no supporting point.
    To define a system of generators indeed corresponding to
    the \f$x\f$ axis, one can add the following code which
    inserts the origin of the space as a vertex:
    \code
  gs.insert(vertex(0*x + 0*y));
    \endcode
    Since dimensions are automatically adjusted, the following
    code obtains the same effect:
    \code
  gs.insert(vertex(0*x));
    \endcode
    In contrast, if we had added the following code, we would have
    defined a line parallel to the \f$x\f$ axis and including
    the point \f$(0, 1)^\transpose \in \Rset^2\f$.
    \code
  gs.insert(vertex(0*x + 1*y));
    \endcode

    \par Example 2
    The following code builds a ray having the same direction as
    the positive part of the \f$x\f$ axis in \f$\Rset^2\f$:
    \code
  GenSys gs;
  gs.insert(ray(x + 0*y));
    \endcode
    To define a system of generators indeed corresponding to the set
    \f[
      \bigl\{\,
        (x, 0)^\transpose \in \Rset^2
      \bigm|
        x \geq 0
      \,\bigr\},
    \f]
    one just has to add the origin:
    \code
  gs.insert(vertex(0*x + 0*y));
    \endcode

    \par Example 3
    The following code builds a system of generators having four vertices
    and corresponding to a square in \f$\Rset^2\f$
    (the same as Example 1 for the system of constraints):
    \code
  GenSys gs;
  gs.insert(vertex(0*x + 0*y));
  gs.insert(vertex(0*x + 3*y));
  gs.insert(vertex(3*x + 0*y));
  gs.insert(vertex(3*x + 3*y));
    \endcode

    \par Example 4
    The following code builds a system of generators having two vertices
    and a ray, corresponding to a half-strip in \f$\Rset^2\f$
    (the same as Example 2 for the system of constraints):
    \code
  GenSys gs;
  gs.insert(vertex(0*x + 0*y));
  gs.insert(vertex(0*x + 1*y));
  gs.insert(ray(x - y));
    \endcode

    \note
    After inserting a multiset of generators in a generator system,
    there are no guarantees that an <EM>exact</EM> copy of them
    can be retrieved:
    in general, only an <EM>equivalent</EM> generator system
    will be available, where original generators may have been
    reordered, removed (if they are duplicate or redundant), etc.
*/
class Parma_Polyhedra_Library::GenSys : private Matrix {
public:
  //! Default constructor: builds an empty system of generators.
  GenSys();
  //! Builds the singleton system containing only generator \p g.
  GenSys(const Generator& g);
  //! Ordinary copy-constructor.
  GenSys(const GenSys& gs);
  //! Destructor.
  virtual ~GenSys();

  //! Assignment operator.
  GenSys& operator =(const GenSys& y);

  //! Returns the dimension of the vector space enclosing \p *this.
  size_t space_dimension() const;

  //! Inserts a copy of the generator \p g into \p *this,
  //! increasing the number of dimensions if needed.
  void insert(const Generator& g);

  //! Returns the singleton system containing only
  //! Generator::zero_dim_vertex().
  static const GenSys& zero_dim_univ();

  /*!
      A const_iterator is used to provide read-only access
      to each generator contained in an object of GenSys.

      \par Example
      The following code prints the system of generators
      of the polyhedron <CODE>ph</CODE>:
      \code
  const GenSys gs = ph.generators();
  GenSys::const_iterator iend = gs.end();
  for (GenSys::const_iterator i = gs.begin(); i != iend; ++i)
    cout << *i << endl;
      \endcode
      The same effect can be obtained more concisely by using
      more features of the STL:
      \code
  const GenSys gs = ph.generators();
  copy(gs.begin(), gs.end(), ostream_iterator<Generator>(cout, "\n"));
      \endcode
  */
  class const_iterator
    : public std::iterator<std::forward_iterator_tag,
				Generator,
				void,
				const Generator*,
                                const Generator&> {

  private:
    Matrix::const_iterator i;

    //! Copy-constructor.
    const_iterator(const Matrix::const_iterator& iter);

    friend class GenSys;

  public:
    //! Default constructor.
    const_iterator();
    //! Ordinary copy-constructor.
    const_iterator(const const_iterator& y);
    //! Destructor.
    virtual ~const_iterator();
    //! Assignment operator.
    const_iterator& operator =(const const_iterator& y);
    //! Dereference operator.
    const Generator& operator *() const;
    //! Indirect member selector.
    const Generator* operator ->() const;
    //! Prefix increment operator.
    const_iterator& operator ++();
    //! Postfix increment operator.
    const_iterator operator ++(int);
    //! Returns <CODE>true</CODE> if and only if
    //! \p *this and \p y are identical.
    bool operator ==(const const_iterator& y) const;
    //! Returns <CODE>true</CODE> if and only if
    //! \p *this and \p y are different.
    bool operator !=(const const_iterator& y) const;
  };

  //! Returns the const_iterator pointing to the first generator,
  //! if \p *this is not empty;
  //! otherwise, returns the past-the-end const_iterator.
  const_iterator begin() const;
  //! Returns the past-the-end const_iterator.
  const_iterator end() const;

private:
  //! Constructor: it builds a system of \p num_rows rays/vertices
  //! on a \p num_columns - 1 dimensional space.
  GenSys(size_t num_rows, size_t num_columns);

  //! Returns the \p k- th generator of the system.
  Generator& operator [](size_t k);
  //! Returns a constant reference to the \p k- th generator of the system.
  const Generator& operator [](size_t k) const;

  //! Checks if the given constraint is satisfied by all generators
  //! in the system.
  GenSys_Con_Rel satisfy(const Constraint& c) const;

  //! Assigns to a given variable an affine expression.
  void affine_image(size_t v,
		    const LinExpression& expr,
		    const Integer& denominator);
  //! Returns the number of lines of the system.
  size_t num_lines() const;
  //! Returns the number of rays of the system.
  size_t num_rays() const;

  //! Removes all the invalid lines and rays (i.e., those with all
  //! the homogeneous terms set to zero).
  void remove_invalid_lines_and_rays();

  //! Checks if all the invariants are satisfied.
  bool OK() const;

private:
  //! Input operator.
  void get(std::istream& s);
  //! Output operator.
  void print(std::ostream& s) const;
};

namespace std {

//! Specialize std::swap.
void swap(Parma_Polyhedra_Library::GenSys& x,
	  Parma_Polyhedra_Library::GenSys& y);

} // namespace std

#include "GenSys.inlines.hh"

#endif
/* Polyhedron class declaration.
   Copyright (C) 2001 Roberto Bagnara <bagnara@cs.unipr.it>

This file is part of the Parma Polyhedra Library (PPL).

The PPL is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

The PPL is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
USA.

For the most up-to-date information see the Parma Polyhedra Library
site: http://www.cs.unipr.it/ppl/ . */

#ifndef _Polyhedron_defs_hh
#define _Polyhedron_defs_hh 1

#include "Variable.defs.hh"
#include "LinExpression.defs.hh"
#include "ConSys.defs.hh"
#include "GenSys.defs.hh"
#include "SatMatrix.defs.hh"
#include "Status.defs.hh"
#include "Polyhedron.types.hh"
#include <set>

namespace Parma_Polyhedra_Library {

  //! Returns <CODE>true</CODE> if and only if
  //! \p x and \p y are the same polyhedron.
  //! \exception std::invalid_argument thrown if \p x and \p y
  //!                                  are dimension-incompatible.
  /*! \relates Polyhedron */
  bool operator ==(const Polyhedron& x, const Polyhedron& y);
  //! Returns <CODE>true</CODE> if and only if
  //! \p x and \p y are different polyhedra.
  //! \exception std::invalid_argument thrown if \p x and \p y
  //!                                  are dimension-incompatible.
  /*! \relates Polyhedron */
  bool operator !=(const Polyhedron& x, const Polyhedron& y);
  //! Returns <CODE>true</CODE> if and only if
  //! \p x is strictly contained in \p y.
  //! \exception std::invalid_argument thrown if \p x and \p y
  //!                                  are dimension-incompatible.
  /*! \relates Polyhedron */
  bool operator <(const Polyhedron& x, const Polyhedron& y);
  //! Returns <CODE>true</CODE> if and only if
  //! \p x strictly contains \p y.
  //! \exception std::invalid_argument thrown if \p x and \p y
  //!                                  are dimension-incompatible.
  /*! \relates Polyhedron */
  bool operator >(const Polyhedron& x, const Polyhedron& y);
  //! Returns <CODE>true</CODE> if and only if
  //! \p x contains \p y.
  //! \exception std::invalid_argument thrown if \p x and \p y
  //!                                  are dimension-incompatible.
  /*! \relates Polyhedron */
  bool operator >=(const Polyhedron& x, const Polyhedron& y);

  // Put them in the namespace here to declare them friend later.
  bool operator <=(const Polyhedron& x, const Polyhedron& y);
  std::ostream& operator <<(std::ostream& s, const Polyhedron& p);
  std::istream& operator >>(std::istream& s, Polyhedron& p);

} // namespace Parma_Polyhedra_Library


//! A convex polyhedron.
/*!
    An object of the class Polyhedron represents a convex polyhedron
    in the vector space \f$\Rset^n\f$.

    The dimension \f$n \in \Nset\f$ of the enclosing vector space
    is a key attribute of the polyhedron:
    - all polyhedra, the empty ones included, are endowed with
      a specific space dimension;
    - most operations working on a polyhedron and another object
      (i.e., another polyhedron, a constraint or generator,
      a set of variables, etc.) will throw an exception if
      the polyhedron and the object are dimension-incompatible
      (see the dimension-compatibility rules in the Introduction);
    - the only ways to change the space dimension of a polyhedron are:
      - <EM>explicit</EM> calls to operators provided for that purpose;
      - standard copy, assignment and swap operators.

    Note that two polyhedra can be defined on the zero-dimension space:
    the empty polyhedron and the universe polyhedron \f$R^0\f$.

    A polyhedron can be specified as either a finite system of constraints
    or a finite system of generators
    (see Minkowski's theorem in the Introduction)
    and it is always possible to obtain either representation.
    That is, if we know the system of constraints, we can obtain
    from this the system of generators that define the same polyhedron
    and vice versa.
    These systems can contain redundant members: in this case we say
    that they are not in the minimal form.

    \par
    In all the examples it is assumed that variables
    <CODE>x</CODE> and <CODE>y</CODE> are defined (where they are
    used) as follows:
    \code
  Variable x(0);
  Variable y(1);
    \endcode

    \par Example 1
    The following code builds a polyhedron corresponding to
    a square in \f$\Rset^2\f$, given as a system of constraints:
    \code
  ConSys cs;
  cs.insert(x >= 0);
  cs.insert(x <= 3);
  cs.insert(y >= 0);
  cs.insert(y <= 3);
  Polyhedron ph(cs);
    \endcode
    The following code builds the same polyhedron as above,
    but starting from a system of generators specifying
    the four vertices of the square:
    \code
  GenSys gs;
  gs.insert(vertex(0*x + 0*y));
  gs.insert(vertex(0*x + 3*y));
  gs.insert(vertex(3*x + 0*y));
  gs.insert(vertex(3*x + 3*y));
  Polyhedron ph(gs);
    \endcode

    \par Example 2
    The following code builds an unbounded polyhedron
    corresponding to a half-strip in \f$\Rset^2\f$,
    given as a system of constraints:
    \code
  ConSys cs;
  cs.insert(x >= 0);
  cs.insert(x - y <= 0);
  cs.insert(x - y + 1 >= 0);
  Polyhedron ph(cs);
    \endcode
    The following code builds the same polyhedron as above,
    but starting from the system of generators specifying
    the two vertices of the polyhedron and one ray:
    \code
  GenSys gs;
  gs.insert(vertex(0*x + 0*y));
  gs.insert(vertex(0*x + y));
  gs.insert(ray(x - y));
  Polyhedron ph(gs);
    \endcode

    \par Example 3
    The following code builds the polyhedron corresponding to
    an half-plane by adding a single constraint
    to the universe polyhedron in \f$\Rset^2\f$:
    \code
  Polyhedron ph(2);
  ph.insert(y >= 0);
    \endcode
    The following code builds the same polyhedron as above,
    but starting from the empty polyhedron in the space \f$\Rset^2\f$
    and inserting the appropriate generators
    (a vertex, a ray and a line).
    \code
  Polyhedron ph(2, Polyhedron::EMPTY);
  ph.insert(vertex(0*x + 0*y));
  ph.insert(ray(y));
  ph.insert(line(x));
    \endcode
    Note that, even if the above polyhedron has no ``proper'' vertex,
    we must add one, because otherwise the result of the Minkowsky's sum
    would be an empty polyhedron.
    To avoid subtle errors related to the minimization process,
    it is required that the first generator inserted in an empty
    polyhedron is a vertex (otherwise, an exception is thrown).

    \par Example 4
    The following code shows the use of the function
    <CODE>add_dimensions_and_embed</CODE>:
    \code
  Polyhedron ph(1);
  ph.insert(x == 2);
  ph.add_dimensions_and_embed(1);
    \endcode
    We build the universe polyhedron in the 1-dimension space \f$\Rset\f$.
    Then we add a single equality constraint,
    thus obtaining the polyhedron corresponding to the singleton set
    \f$\{ 2 \} \sseq \Rset\f$.
    After the last line of code, the resulting polyhedron is
    \f[
      \bigl\{\,
        (2, x_1)^\transpose \in \Rset^2
      \bigm|
        x_1 \in \Rset
      \,\bigr\}.
    \f]

    \par Example 5
    The following code shows the use of the function
    <CODE>add_dimensions_and_project</CODE>:
    \code
  Polyhedron ph(1);
  ph.insert(x == 2);
  ph.add_dimensions_and_project(1);
    \endcode
    The first two lines of code are the same as in Example 4 for
    <CODE>add_dimensions_and_embed</CODE>.
    After the last line of code, the resulting polyhedron is
    the singleton set
    \f$\bigl\{ (2, 0)^\transpose \bigr\} \sseq \Rset^2\f$.

    \par Example 6
    The following code shows the use of the function
    <CODE>affine_image</CODE>:
    \code
  Polyhedron ph(2, Polyhedron::EMPTY);
  ph.insert(vertex(0*x + 0*y));
  ph.insert(vertex(0*x + 3*y));
  ph.insert(vertex(3*x + 0*y));
  ph.insert(vertex(3*x + 3*y));
  LinExpression coeff = x + 4;
  ph.affine_image(x, coeff);
    \endcode
    In this example the starting polyhedron is a square in \f$\Rset^2\f$,
    the considered variable is \f$x\f$ and the affine expression is \f$x+4\f$.
    The resulting polyhedron is the same square translated towards right.
    Moreover, if the affine transformation for the same variable \p x
    is \f$x+y\f$:
    \code
  LinExpression coeff = x + y;
    \endcode
    the resulting polyhedron is a parallelogram with the height equal to
    the side of the square and the oblique sides parallel to the line
    \f$x-y\f$.
    Instead, if we do not use an invertible transformation for the same
    variable; for example, the affine expression \f$y\f$:
    \code
  LinExpression coeff = y;
    \endcode
    the resulting polyhedron is a diagonal of the square.

    \par Example 7
    The following code shows the use of the function
    <CODE>affine_preimage</CODE>:
    \code
  Polyhedron ph(2);
  ph.insert(x >= 0);
  ph.insert(x <= 3);
  ph.insert(y >= 0);
  ph.insert(y <= 3);
  LinExpression coeff = x + 4;
  ph.affine_preimage(x, coeff);
    \endcode
    In this example the starting polyhedron, \p var and the affine
    expression and the denominator are the same as in Example 6,
    while the resulting polyhedron is again the same square,
    but translated towards left.
    Moreover, if the affine transformation for \p x is \f$x+y\f$
    \code
  LinExpression coeff = x + y;
    \endcode
    the resulting polyhedron is a parallelogram with the height equal to
    the side of the square and the oblique sides parallel to the line
    \f$x+y\f$.
    Instead, if we do not use an invertible transformation for the same
    variable \p x, for example, the affine expression \f$y\f$:
    \code
  LinExpression coeff = y;
    \endcode
    the resulting polyhedron is a line that corresponds to the \f$y\f$ axis.

    \par Example 8
    For this example we use also the variables:
    \code
  Variable z(2);
  Variable w(3);
    \endcode
    The following code shows the use of the function
    <CODE>remove_dimensions</CODE>:
    \code
  GenSys gs;
  gs.insert(vertex(3*x + y +0*z + 2*w));
  Polyhedron ph(gs);
  set<Variable> to_be_removed;
  to_be_removed.insert(y);
  to_be_removed.insert(z);
  ph.remove_dimensions(to_be_removed);
    \endcode
    The starting polyhedron is the singleton set
    \f$\bigl\{ (3, 1, 0, 2)^\transpose \bigr\} \sseq \Rset^4\f$, while
    the resulting polyhedron is
    \f$\bigl\{ (3, 2)^\transpose \bigr\} \sseq \Rset^2\f$.
    Be careful when removing dimensions <EM>incrementally</EM>:
    since dimensions are automatically renamed after each application
    of the <CODE>remove_dimensions</CODE> operator, unexpected results
    can be obtained.
    For instance, by using the following code we would obtain
    a different result:
    \code
  set<Variable> to_be_removed1;
  to_be_removed1.insert(y);
  ph.remove_dimensions(to_be_removed1);
  set<Variable> to_be_removed2;
  to_be_removed2.insert(z);
  ph.remove_dimensions(to_be_removed2);
    \endcode
    In this case, the result is the polyhedron
    \f$\bigl\{(3, 0)^\transpose \bigr\} \sseq \Rset^2\f$:
    when removing the set of dimensions \p to_be_removed2
    we are actually removing variable \f$w\f$ of the original polyhedron.
    For the same reason, the operator \p remove_dimensions
    is not idempotent: removing twice the same set of dimensions
    is never a no-op.
*/

class Parma_Polyhedra_Library::Polyhedron {
public:
  //! Kinds of degenerate polyhedra.
  enum Degenerate_Kind {
    //! The universe polyhedron, i.e., the whole vector space.
    UNIVERSE,
    //! The empty polyhedron, i.e., the empty set.
    EMPTY
  };

  //! Ordinary copy-constructor.
  Polyhedron(const Polyhedron& y);
  //! Builds either the universe or the empty polyhedron of dimension
  //! \p num_dimensions. Both parameters are optional:
  //! by default, a 0-dimension space universe polyhedron is built.
  explicit Polyhedron(size_t num_dimensions = 0,
		      Degenerate_Kind kind = UNIVERSE);
  //! Builds a polyhedron from a system of constraints.
  //! The polyhedron inherits the space dimension of the constraint system.
  //! \param cs       The system of constraints defining the polyhedron.
  //!                 It is not declared <CODE>const</CODE>
  //!                 because it can be modified.
  Polyhedron(ConSys& cs);
  //! Builds a polyhedron from a system of generators.
  //! The polyhedron inherits the space dimension of the generator system.
  //! \param gs       The system of generators defining the polyhedron.
  //!                 It is not declared <CODE>const</CODE>
  //!                 because it can be modified.
  //! \exception std::invalid_argument thrown if the system of generators
  //!                                  is not empty but has no vertices.
  Polyhedron(GenSys& gs);
  // Destructor
  ~Polyhedron();

  //! The assignment operator.
  //! (Note that \p *this and \p y can be dimension-incompatible.)
  Polyhedron& operator =(const Polyhedron& y);

  //! Returns the dimension of the vector space enclosing \p *this.
  size_t space_dimension() const;
  //! Intersects \p *this with polyhedron \p y and
  //! assigns the result to \p *this.
  //! \exception std::invalid_argument thrown if \p *this and \p y
  //!                                  are dimension-incompatible.
  void intersection_assign_and_minimize(const Polyhedron& y);
  //! Intersects \p *this with polyhedron \p y and assigns the result
  //! to \p *this without minimizing the result.
  //! \exception std::invalid_argument thrown if \p *this and \p y
  //!                                  are dimension-incompatible.
  void intersection_assign(const Polyhedron& y);
  //! Assigns the convex hull of \p *this \f$\cup\f$ \p y to \p *this.
  //! \exception std::invalid_argument thrown if \p *this and \p y
  //!                                  are dimension-incompatible.
  void convex_hull_assign_and_minimize(const Polyhedron& y);
  //! Assigns the convex hull of \p *this \f$\cup\f$ \p y to \p *this,
  //! without minimizing the result.
  //! \exception std::invalid_argument thrown if \p *this and \p y
  //!                                  are dimension-incompatible.
  void convex_hull_assign(const Polyhedron& y);

  //! Returns the relation between the generators of \p *this
  //! and the constraint \p c.
  //! \exception std::invalid_argument thrown if \p *this and constraint
  //!                                  \p c are dimension-incompatible.
  GenSys_Con_Rel satisfies(const Constraint& c);
  //! Tests the inclusion of the generator \p g in the
  //! polyhedron \p *this.
  //! \exception std::invalid_argument thrown if \p *this and constraint
  //!                                  \p g are dimension-incompatible.
  bool includes(const Generator& g);

  //! Computes the widening between \p *this and \p y and
  //! assigns the result to \p *this.
  //! \param y           The polyhedron that <EM>must</EM>
  //!                    be contained in \p *this.
  //! \exception std::invalid_argument thrown if \p *this and \p y
  //!                                  are dimension-incompatible.
  void widening_assign(const Polyhedron& y);
  //! Limits the widening between \p *this and \p y by \p cs
  //! and assigns the result to \p *this.
  //! \param y                 The polyhedron that <EM>must</EM>
  //!                          be contained in \p *this.
  //! \param cs                The system of constraints that limits
  //!                          the widened polyhedron. It is not
  //!                          declared <CODE>const</CODE>
  //!                          because it can be modified.
  //! \return       <CODE>true</CODE> if the resulting polyhedron is not
  //!               empty <CODE>false</CODE> otherwise.
  //! \exception std::invalid_argument thrown if \p *this, \p y and
  //!                                  \p cs are dimension-incompatible.
  bool limited_widening_assign(const Polyhedron& y, ConSys& cs);

  //! Returns the system of constraints.
  const ConSys& constraints() const;
  //! Returns the system of generators.
  const GenSys& generators() const;

  //! Inserts a copy of constraint \p c into the system of constraints
  //! of \p *this.
  //! \exception std::invalid_argument thrown if \p *this and constraint \p c
  //!                                  are dimension-incompatible.
  void insert(const Constraint& c);

  //! Inserts a copy of generator \p g into the system of generators
  //! of \p *this.
  //! \exception std::invalid_argument thrown if \p *this and generator \p g
  //!                                  are dimension-incompatible
  //!                                  or if a ray/line is inserted
  //!                                  in an empty polyhedron.
  void insert(const Generator& g);

  //! Transforms the polyhedron \p *this, assigning an affine expression
  //! to the specified variable.
  //! \param v             The variable to which the affine
  //!                      expression is assigned.
  //! \param expr          The numerator of the affine expression.
  //! \param denominator   The denominator of the affine expression
  //!                      (optional argument with default value 1.)
  //! \exception std::invalid_argument thrown if \p denominator is zero
  //!                                  or if \p expr and \p *this
  //!                                  are dimension-incompatible
  //!                                  or if \p v is not a dimension
  //!                                  of \p *this.
  void affine_image(const Variable& v,
		    const LinExpression& expr,
		    const Integer& denominator = Integer_one());
  //! Transforms the polyhedrons \p *this, substituting an affine
  //! expression for the specified variable. (It is the inverse
  //! operation of <CODE>affine_image</CODE>.)
  //! \param v             The variable to which the affine expression
  //!                      is substituted.
  //! \param expr          The numerator of the affine expression.
  //! \param denominator   The denominator of the affine expression
  //!                      (optional argument with default value 1.)
  //! \exception std::invalid_argument thrown if \p denominator is zero
  //!                                  or if \p expr and \p *this
  //!                                  are dimension-incompatible
  //!                                  or if \p v is not a dimension
  //!                                  of \p *this.
  void affine_preimage(const Variable& v,
		       const LinExpression& expr,
		       const Integer& denominator = Integer_one());

  //! Checks if all the invariants are satisfied.
  //! \param check_not_empty    <CODE>true</CODE> if it must be checked
  //!                           whether the system of constraint is
  //!                           satisfiable.
  //! \return       <CODE>true</CODE> if the polyhedron satisfies
  //!               all the invariants stated in the PPL,
  //!               <CODE>false</CODE> otherwise.
  bool OK(bool check_not_empty = true) const;

private:
  //! Minimizes generators and constraints.
  void minimize() const;
  //! Updates constraints starting from generators and minimizes them.
  void update_constraints() const;
  //! Updates generators starting from constraints and minimizes them.
  bool update_generators() const;
  //! Updates \p sat_c using the updated constraints and generators.
  void update_sat_c() const;
  //! Updates \p sat_g using the updated constraints and generators.
  void update_sat_g() const;
  //! Sorts the matrix of constraints keeping \p sat_g consistent.
  void obtain_sorted_constraints();
  //! Sorts the matrix of generators keeping \p sat_c consistent.
  void obtain_sorted_generators();
  //! Sorts the matrix of constraints and makes \p sat_c consistent.
  void obtain_sorted_constraints_with_sat_c();
  //! Sorts the matrix of generators and makes \p sat_g consistent.
  void obtain_sorted_generators_with_sat_g();

public:
  //! Adds new dimensions and embeds the old polyhedron into the new space.
  //! \param dim      The number of dimensions to add.
  void add_dimensions_and_embed(size_t dim);
  //! Adds new dimensions to the polyhedron
  //! and does not embed it in the new space.
  //! \param dim      The number of dimensions to add.
  void add_dimensions_and_project(size_t dim);
  //! Removes the specified dimensions.
  //! \param to_be_removed The set of variables to remove.
  void remove_dimensions(const std::set<Variable>& to_be_removed);
  //! Removes all dimensions higher than a threshold.
  //! \param new_dimension The dimension of the resulting polyhedron
  //!                      after all higher dimensions have been removed.
  void remove_higher_dimensions(size_t new_dimension);
  //! Adds the specified constraints and computes a new polyhedron.
  //! \param  cs            The constraints that will be added to the
  //!                       current system of constraints. This parameter
  //!                       is not declared <CODE>const</CODE> because
  //!                       it can be modified.
  //! \return               <CODE>false</CODE> if the resulting
  //!                       polyhedron is empty.
  //! \exception std::invalid_argument thrown if \p *this and \p cs
  //!                                  are dimension-incompatible.
  bool add_constraints_and_minimize(ConSys& cs);
  //! Adds the specified constraints without minimizing.
  //! \param  cs             The constraints that will be added to the
  //!                        current system of constraints. This parameter
  //!                        is not declared <CODE>const</CODE> because
  //!                        it can be modified.
  //! \exception std::invalid_argument thrown if \p *this and \p cs
  //!                                  are dimension-incompatible.
  void add_constraints(ConSys& cs);

  //! First increases the space dimension of \p *this by adding
  //! \p cs.space_dimension() new dimensions;
  //! then adds to the system of constraints of \p *this
  //! a renamed-apart version of the constraints in `cs'.
  void add_dimensions_and_constraints(ConSys& cs);
  //! Adds the specified generators.
  //! \param  gs          The generators that will be added to the
  //!                     current system of generators. The parameter is
  //!                     not declared <CODE>const</CODE> because it
  //!                     can be modified.
  //! \exception std::invalid_argument thrown if \p *this and
  //!                                  \p gs are dimension-incompatible.
  void add_generators_and_minimize(GenSys& gs);
  //! Adds the specified generators without minimizing.
  //! \param  gs             The generators that will be added to the
  //!                        current system of generators. This parameter
  //!                        is not declared <CODE>const</CODE> because
  //!                        it can be modified.
  //! \exception std::invalid_argument thrown if \p *this and \p gs
  //!                                  are dimension-incompatible.
  void add_generators(GenSys& gs);

  //! Returns <CODE>true</CODE> if and only if \p *this is
  //! an empty polyhedron.
  bool check_empty() const;
  //! Returns <CODE>true</CODE> if and only if \p *this
  //! is a universe polyhedron.
  bool check_universe() const;

  //! Returns <CODE>true</CODE> if and only if
  //! polyhedron \p x is contained in polyhedron \p y.
  //! \exception std::invalid_argument thrown if \p x and \p y
  //!                                  are dimension-incompatible.
  friend bool Parma_Polyhedra_Library::operator <=(const Polyhedron& x,
						   const Polyhedron& y);

  //! Output operator.
  friend std::ostream&
  Parma_Polyhedra_Library::operator <<(std::ostream& s, const Polyhedron& p);

  //! Input operator.
  friend std::istream&
  Parma_Polyhedra_Library::operator >>(std::istream& s, Polyhedron& p);

  //! Swaps \p *this with polyhedron \p y.
  //! (Note that \p *this and \p y can be dimension-incompatible.)
  void swap(Polyhedron& y);

private:
  //! The number of dimensions of the enclosing vector space.
  size_t space_dim;
  //! The system of constraints.
  ConSys con_sys;
  //! The system of generators.
  GenSys gen_sys;
  //! The saturation matrix having constraints on its columns.
  SatMatrix sat_c;
  //! The saturation matrix having generators on its columns.
  SatMatrix sat_g;
  //! The status flags to keep track of the polyhedron's internal state.
  Status status;

public:
  //! Returns <CODE>true</CODE> if and only if
  //! \p *this is an empty polyhedron.
  bool is_empty() const;

private:
  /*! @name Private Verifiers
    Verify if individual flags are set.
  */
  //@{
  bool constraints_are_up_to_date() const;
  bool generators_are_up_to_date() const;
  bool constraints_are_minimized() const;
  bool generators_are_minimized() const;
  bool sat_c_is_up_to_date() const;
  bool sat_g_is_up_to_date() const;
  //@}


  /*! @name State flag setters.
    Set only the specified flags.
  */
  //@{
  void set_zero_dim_univ();
  void set_empty();
  void set_constraints_up_to_date();
  void set_generators_up_to_date();
  void set_constraints_minimized();
  void set_generators_minimized();
  void set_sat_c_up_to_date();
  void set_sat_g_up_to_date();
  //@}

  /*! @name State flag cleaners.
    Clear only the specified flag.
  */
  //@{
  void clear_empty();
  void clear_constraints_up_to_date();
  void clear_generators_up_to_date();
  void clear_constraints_minimized();
  void clear_generators_minimized();
  void clear_sat_c_up_to_date();
  void clear_sat_g_up_to_date();
//@}

  //! Adds new dimensions to the given matrices.
  static void add_dimensions(Matrix& mat1,
                             Matrix& mat2,
                             SatMatrix& sat1,
                             SatMatrix& sat2,
			     size_t add_dim);

  //! Performs the conversion from constraints to generators and vice versa.
  static size_t conversion(Matrix& entry,
			   size_t start,
			   Matrix& result,
			   SatMatrix& sat,
			   size_t num_lines_or_equalities);

  //! Uses Gauss' elimination method to simplify the result of
  //! <CODE>conversion()</CODE>.
  static int simplify(Matrix& mat, SatMatrix& sat);

  //! Builds and simplifies constraints from generators (or vice versa).
  static bool minimize(bool con_to_ray, Matrix& source, Matrix& dest,
		       SatMatrix& sat);

  //! Adds given constraints and builds minimized corresponding generators
  //! or vice versa.
  static bool add_and_minimize(bool con_to_ray,
			       Matrix& source1, Matrix& dest, SatMatrix& sat,
			       const Matrix& source2);
};

namespace std {
  //! Specialize std::swap.
  void swap(Parma_Polyhedra_Library::Polyhedron& x,
	    Parma_Polyhedra_Library::Polyhedron& y);
}

#include "Polyhedron.inlines.hh"

#endif
